#!/usr/bin/env python2.3

# We combine the use of Platypus and the database
# to print the db data nicely

import os,sys

# Load the bits of Platypus that we'll be using
from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, Table, TableStyle

# And import a few utility variables
from reportlab.lib.units import cm
from reportlab.lib.pagesizes import A4
from reportlab.lib import colors

# Load the database interface
from pyPgSQL import PgSQL

# Our DB query. Returns two columns.
query = """
SELECT DISTINCT
    id,
    customer_code,
    issue_week,
    first_run_week,
    last_run_week
FROM booking
WHERE     issue_week < '2004-02-01'
      AND issue_week > '2004-01-01'
ORDER BY customer_code, issue_week
"""

# Have we been passed exactly one argument (remember, sys.argv[0]
# is the executable name)?
if len(sys.argv) == 2:
    # Set the output file name to the first command line argument
    outfile = sys.argv[1]
else:
    # Exit messily, returning a usage message. Normally you'd
    # use print and sys.exit() instead, but we don't care right
    # now if the user sees a traceback.
    raise Exception,"Usage: %s output_file.pdf" % sys.argv[0]

# Create the Platypus document instance we'll be using to draw
# everything and create the PDF. We don't need to do much to it
# in this case. Note that we pass a file _name_, a simple string,
# here. We could pass a file object, too, and ReportLab would
# figure it out.
# We're letting the document template make all the layout decisions
# in this simple example.
doc = SimpleDocTemplate(outfile)

# Make an array to hold the objects to be rendered onto 
# the canvas
elements = []

# connect to the DB using default parameters. With PostgreSQL
# this will use the local UNIX socket, performing authentication
# by local user ID.
conn = PgSQL.connect()

# Create a cursor on the connection. Cursors are what we actually
# use to execute the query and retrieve results.
curs = conn.cursor()

# run the query. We're cheating and hardcoding the dates into the
# query to avoid bringing date calculations into the program, so we
# don't need to pass any params to be substituted.
# in this case, we skip any entries with NULL "last_run_week" values.
# (Yes, I know we we could do that in the query; that's not the point).
curs.execute(query)

# Make another array to hold the data we plan to put
# in the table. We pre-fill it with the first row,
# which contains the column headings.
table_data = [["Booking No.", "cCode", "Issue Week", "First Week", "Last Week"]]

# Iterate over the "input" data, adding it to the dict
# of data to be sent to the table renderer.
# You might do formatting, checking, etc on the row
# data as you read it inside this loop.
for row in curs.fetchall():
    if not row.last_run_week:
        continue
    table_data.append( [row.id,
                        row.customer_code,
                        row.issue_week.Format("%F"),
                        row.first_run_week.Format("%F"),
                        row.last_run_week.Format("%F"),
                        ])

# Construct the Platypus table object that'll render it all,
# feeding it the table_data array. We say nothing about
# formatting, so it'll make its own decisions - usually effective,
# but unimpressive.
t = Table(table_data,
          repeatRows=1      # This causes it to repeat the headings
                            # on each new page.
          )

# Add the table to the list of elements we'll be asking Platypus
# to draw onto the page.
elements.append(t)

# Render the page to the canvas and save it.
doc.build(elements)
