#!/usr/bin/env python2.3
#
# This program prints a nice report of total bookings per customer in a given
# month to a nicely formatted PDF.
#
# Also supports use as a CGI program.
#

import os,sys

from mx import DateTime
from pyPgSQL import PgSQL,libpq

from reportlab.platypus import SimpleDocTemplate, Paragraph, Spacer, Table, TableStyle
from reportlab.lib.units import cm
from reportlab.lib.pagesizes import A4
from reportlab.lib import colors

# The query used to actually grab the data out of the DB.
# Dates are substituted into it later.
query = """
SELECT
    customer.code,
    customer.name,
    SUM(orig_amount) AS post_gst,
    SUM(orig_amount) / 1.1 AS pre_gst,
    COUNT(trans_type) AS num
FROM ledger INNER JOIN customer ON (customer.code = ledger.customer_code)
WHERE
    ledger.trans_type = 'I'
    AND ledger.trans_date >= %(start_date)s
    AND ledger.trans_date < %(end_date)s
GROUP BY
    customer.code,
    customer.name
ORDER BY customer.code;
"""

def cgiCheck():
    """Check to see if we're running as a CGI script by checking for the
    QUERY_STRING env var. If found, set ourselves up to run as a CGI program."""

    global form,iscgi

    if 'QUERY_STRING' in os.environ:
        iscgi = True
        # Yup, we're running as a CGI program. Load CGI support.
        import cgi
        import cgitb; cgitb.enable()
        form = cgi.FieldStorage()
    else:
        # Nope, looks like we're producing a PDF file instead.
        iscgi = False

def dbConnect():
    """Connect to the database and run our query"""
    global dbd,conn,cu
    # Connect to the DB
    dbd = PgSQL
    conn = dbd.connect(database='craig',user='craig')
    cu = conn.cursor()
    # The query - one line in the noise
    cu.execute(query, q_data)

def drawFirstPageHeading(canvas):
    """Draw the large, double-underlined heading used on the first page"""
    canvas.saveState()

    # Draw page 1 heading
    canvas.setFont("Times-Bold",18)
    headbase = PAGE_HEIGHT - MARGIN - 18
    canvas.drawCentredString(PAGE_WIDTH / 2, headbase , TITLE)

    # double underline page 1 heading - involves depressingly
    # large amounts of calculations. Must be a simpler way to do it
    # - there always is.
    headwidth = canvas.stringWidth(TITLE, "Times-Bold", 18)
    (line1base, line2base) = (headbase- 0.05*cm, headbase - 0.10*cm)
    leftend = (PAGE_WIDTH - headwidth) / 2
    rightend = PAGE_WIDTH - leftend
    canvas.setLineWidth(0.02*cm)
    canvas.line( leftend , line1base, rightend, line1base )
    canvas.line( leftend , line2base, rightend, line2base )

    canvas.restoreState()

def firstPage(canvas,doc):
    """This function is passed to the the build method of the document
    and is used to set up some forms, draw a heading and draw the footer."""
    canvas.saveState()
    setupFooter(canvas)
    drawFirstPageHeading(canvas)
    canvas.setFont("Times-Roman",8)
    canvas.drawRightString(PAGE_WIDTH - MARGIN, MARGIN, "Page %s" % doc.page)
    canvas.doForm("Footer")
    canvas.restoreState()

def otherPages(canvas,doc):
    """Draws the footer from a predefined form, and adds a page number."""
    canvas.saveState()
    # Note that for static information we simply re-use
    # the form we defined earlier. Almost no data is added
    # to the PDF to do this.
    canvas.doForm("Footer")
    canvas.setFont("Times-Roman",8)
    canvas.drawRightString(PAGE_WIDTH - MARGIN, MARGIN, "Page %s" % doc.page)
    canvas.restoreState()

def setupFooter(canvas):
    """Create a PDF form to hold the static footer information"""
    canvas.beginForm("Footer")
    # insert a date stamp
    canvas.setFont("Helvetica-BoldOblique",8)
    datestamp = "Printed %s" % DateTime.now().Format("%F %H:%M:%S")
    canvas.drawString(MARGIN, MARGIN, datestamp)
    # and short report title
    canvas.setFont("Times-Bold",8)
    canvas.drawCentredString( PAGE_WIDTH / 2, MARGIN, SHORT_TITLE )
    canvas.endForm()

def main():
    global PAGE_WIDTH, PAGE_HEIGHT, MARGIN
    PAGE_WIDTH, PAGE_HEIGHT = A4
    MARGIN = 0.5*cm

    # Running as a CGI script?
    cgiCheck()

    # check arguments
    if iscgi:
        outfile = sys.stdout
        year = int(form.getvalue("year","").strip())
        month = int(form.getvalue("month","").strip())
        idate = DateTime.Date(year,month,1)
    elif len(sys.argv) < 3:
        print "Usage: %s <iso-date> output\n   eg %s 2004-04-01 report.pdf" % (sys.argv[0], sys.argv[0])
        sys.exit()
    else:
        idate = sys.argv[1].strip()
        outfilename = sys.argv[2].strip()
        if not outfilename.lower().endswith('.pdf'):
            raise Exception,"Unrecognised format ext on filename. Supported formats are 'pdf' and 'txt'"
        outfile = file(outfilename,"w")
        idate = DateTime.Date( int(idate[0:4]), int(idate[5:7]), int(idate[8:10]) )

    # Figure out the date range to work on based on what we were passed
    global q_data
    q_data = {}
    q_data['start_date']  = DateTime.Date( idate.year, idate.month,     01 )
    q_data['end_date']    = DateTime.Date( idate.year, idate.month + 1, 01 )

    global TITLE, SHORT_TITLE, AUTHOR
    TITLE       = "Per-Customer Monthly Booking Totals For %s\n\n" % q_data['start_date'].Format("%h %Y")
    SHORT_TITLE = "Monthly Customer Totals - %s" % q_data['start_date'].Format("%h %Y")
    AUTHOR      = "Craig Ringer, POST Newspapers"

    dbConnect()

    doc = SimpleDocTemplate(outfile)
    halfcm = 0.5*cm
    # The default 1 inch margins are excessive; reduce them.
    doc.leftMargin, doc.rightMargin, = halfcm, halfcm
    doc.topMargin, doc.bottomMargin  = halfcm, halfcm

    # This is the list of elements to include in the document. The spacer is
    # included to push the table down on the first page, making room for the
    # report title. The title is drawn directly onto the canvas, not using
    # Platypus.
    elements = [Spacer(1,1.5*cm)]
    #style = styles["Normal"]

    # Table_data is, much like elements, a list of objects to go in the table.
    # in this case, however, it's a list of rows, each row being defined by a list.
    # We define the headings row initially.
    table_data = [ ['cCode', 'Customer Name', 'Inc. GST', 'Pre GST', 'Num'] ]

    # Define the styling information for the table. Later rules override newer ones
    # when there's a conflict, which is useful for (eg) font settings.
    table_style = TableStyle()
    # underline the headings
    table_style.add('LINEBELOW', (0,0), (-1,0), 1, colors.black)
    # Column headings should be in courier bold
    table_style.add('FONTNAME', (0,0), (-1,0), "Courier-Bold")
    # Except customer name, which is in Helvetica Bold
    table_style.add('FONTNAME', (1,0), (1,0), "Helvetica-Bold")
    # Numeric columns should be printed in courier
    table_style.add('FONTNAME', (2,1), (-1,-1), "Courier")
    # Customer name should appear in Helvetica
    table_style.add('FONTNAME', (1,1), (1,-1), "Helvetica")
    # Customer code should appear in Courier Bold
    table_style.add('FONTNAME', (0,1), (0,-1), "Courier-Bold")

    try:
        # Fire up the JIT compiler now we're in the meat of the app
        import psyco
        psyco.full()
    except:
        pass

    # Iterate over DB output here
    for row in cu.fetchall():
        table_data.append([
                "%6s" % row.code,
                "%-28s" % str(row.name).title(),
                "%8.2f" % row.post_gst,
                "%8.2f" % row.pre_gst,
                row.num
                ])

    # Add the table to the list of elements to be drawn
    t = Table(table_data,style=table_style,repeatRows=1)
    elements.append(t)

    # If we're running as a CGI program, print some headers
    if iscgi:
        print "Content-Type: application/pdf"
        print "Content-Disposition: attachment; filename=report.pdf"
        print
    # Now actually format and output the document
    doc.build(elements, onFirstPage=firstPage, onLaterPages=otherPages)

if __name__ == '__main__': main()
